#
# -*- coding: utf-8 -*-
#
# select_font.py

#フォントファミリーの一覧を取得
#リストボックスでフォントを選択
#決定前の選択状態のときにサンプル文字列を縦書き表示
#決定でフォントを選択

from tkinter import *
from tkinter import simpledialog
from tkinter import font
import platform
import zenhan
import re

#グローバル変数
plsystem = platform.system()
if plsystem == 'Windows':
  dlgfntname = 'ＭＳ ゴシック'
else:
  dlgfntname = 'IPA明朝'

dlgfont1 = (dlgfntname,'10','bold')
dlgfont2 = (dlgfntname,'8')

fontlist = []
smpdatestr = '平成29年3月31日'
smpaddrstr = 'あの県あの市とある町\n大字某２３４５\nABCマンション２３号'
smpaddrstr2 = '##あの県あの市とある町\n大字某２３４５\nABCマンション２３号'
fontsizestr = '12'
fontsize = 12
#DPI(dot_per_inch)標準
dpi = 96
#Point_per_inch（固定）１ポイントは1/72インチ
ppi = 72
fntsp = 3
canvaswd = 200
datecoords = [int(canvaswd/6),10]
addrcoords2 = [int(canvaswd*3/6),10]
addrcoords = [int(canvaswd*5/6),10]

#-----------------------------------------
#インストール済みのフォントリストを返す
#重複したフォント名を排除
#-----------------------------------------
def get_font_list():
  flist = []
  for fnt in font.families():
    if not fnt in flist:
      flist.append(fnt)
  return flist

#------------------------------------------
#フォントリストに指定のフォントが含まれるか否かを返す
#------------------------------------------
def is_font_insystem(tgtfnt,fntlist=None):
  if fntlist == None:
    fntlist = get_font_list()
  if tgtfnt in fntlist:
    return True
  else:
    return False

#---------------------------------------------------------
#リストボックスのあるダイアログ-------class FontSelect_dialog
#---------------------------------------------------------
#タイトルは呼び出し時に引数で指定する
class FontSelect_dialog(simpledialog.Dialog):
  def body(self,master):
    #承継元のアトリビュートを上書き
    self.result = []
    #選択行
    self.cur_idx = None
    #選択中のフォント名
    self.fontname = None
    #フォントタプル
    self.fonttpl = None
    #ラベル
    msg = '表示用フォントに設定するフォントを選んでください。'
    Label(master,text=msg,font=dlgfont1).grid(padx=2,pady=2,row=0,columnspan=3)
    Label(master,text='フォント名').grid(padx=2,pady=2,row=1)
    Label(master,text='サンプル').grid(padx=2,pady=2,row=1,column=2)

    self.yscroll = Scrollbar(master, orient=VERTICAL)
    self.yscroll.grid(row=2, column=1, sticky=N+S)

    #フォント名選択リストボックス
    self.listbox = Listbox(master,width=30,height=15,font='Times 10',
                           selectmode=BROWSE,
                           yscrollcommand=self.yscroll.set)
    self.listbox.grid(padx=2,pady=2,row=2,column=0,sticky=W)
    self.yscroll['command'] = self.listbox.yview
    for fnt in fontlist:
      self.listbox.insert(END,fnt)
    #注意書き
    msg='※日本語フォント以外のサンプル表示はシステムが\n'
    msg += '代替フォントで表示している可能性があります。'
    Label(master,text=msg,font=dlgfont2).grid(row=3,columnspan=3,sticky=W+E)
    #サンプル表示用フレーム
    dispframe = Frame(master,relief=SUNKEN)
    dispframe.configure(borderwidth=2)
    dispframe.grid(padx=2,pady=2,row=2,column=2)
    #フレーム上のサンプル表示用キャンバス
    self.smpcanvas = Canvas(dispframe,width=canvaswd)
    self.smpcanvas.grid(sticky=N+S)
    #フォーカス
    self.listbox.focus_set()
    self.check_selection()

  def apply(self):
    self.cur_idx = self.listbox.curselection()
    if len(self.cur_idx):
      self.result = self.listbox.get(self.cur_idx[0])

  #250ms毎に選択状況を調べる
  def check_selection(self):
    cur = self.listbox.curselection()
    if cur != self.cur_idx:
      self.cur_idx = cur
      if len(self.cur_idx):
        #print(self.listbox.get(self.cur_idx[0]))
        self.fontname = self.listbox.get(self.cur_idx[0])
        self.sample_show()
    self.after(250,self.check_selection)

  #smpcanvasに住所と日付のサンプル文字列を表示
  def sample_show(self):
    smptatedate = tatedatestr(smpdatestr)
    #smptateaddr = zentategaki(smpaddrstr)
    smptateaddr = memotatenaka(smpaddrstr)
    smptateaddr2 = memotatenaka(smpaddrstr2)
    self.fonttpl=(self.fontname,fontsizestr)
    #以前の表示を消去
    self.smpcanvas.delete('sample')
    #縦書テキストのアンカーを使う
    #年月日
    lst= smptatedate.split("\n")
    x = datecoords[0]
    self.dateshow(lst,x,fontsize,self.fonttpl)
    #住所
    fh = int(fontsize * dpi / ppi)
    y0 = addrcoords[1]+fntsp
    x = addrcoords[0]+fntsp
    for lad in smptateaddr:
      y=y0
      for tx in lad.split():
        self.smpcanvas.create_text(x,y,text=tx,font=self.fonttpl,
                                   anchor='n',tags='sample')
        y += fh
      x -= fh
    #住所2
    fh = int(fontsize * dpi / ppi)
    y0 = addrcoords2[1]+fntsp
    x = addrcoords2[0]+fntsp
    for lad in smptateaddr2:
      y=y0
      for tx in lad.split():
        self.smpcanvas.create_text(x,y,text=tx,font=self.fonttpl,
                                   anchor='n',tags='sample')
        y += fh
      x -= fh

##    tl = smptateaddr.split("\n")
##    x = addrcoords[0]+fntsp
##    y = addrcoords[1]+fntsp
##    fh = int(fontsize * dpi / ppi)
##    for tlt in tl:
##      self.smpcanvas.create_text(x,y,text=tlt,font=self.fonttpl,
##                                 anchor='nw',tags='sample')
##      y += fh
      
  def dateshow(self,tl,x,size,fnt):
    imx = 0
    for i in tl:
      icnt = len(i)
      imx = icnt if icnt>imx else imx
    fn = int(size * dpi / ppi)
    y = datecoords[1]
##    if imx == 4:   #西暦年の可能性
##      #y = datecoords[1]-(fn*(len(tl)+1)+fntsp)
##      y = datecoords[1]
##    else:
##      #y = datecoords[1]-(fn*len(tl)+fntsp)
    for tlt in tl:
      if len(tlt)==4:   #西暦年表示
        self.smpcanvas.create_text(x,y,text=tlt[0:2],font=fnt,
                                   anchor='n',tags='sample')
        y += fn
        self.smpcanvas.create_text(x+fntsp,y,text=tlt[2:4],font=fnt,
                                   anchor='n',tags='sample')
        y += fn
      else:
        self.smpcanvas.create_text(x,y,text=tlt,font=fnt,anchor='n',
                                   tags='sample')
        y += fn
    
#------------------------------------- class FontSelect_dialog  end

#----------------------------------------------------------
#与えられた文字列を全角の縦書き表示用の文字列に変換（並替え）
#----------------------------------------------------------
def zentategaki(basestr):
  #前後の空白や空行削除
  #半角→全角（何故か'-'は対象外）
  zen = zenhan.h2z(basestr.strip())

  #数字をつなぐ'ー|一|‐'を'｜'に置き換える。
  #漢数字の一も｜に置き換わるので漢数字の一を使いたい場合は￥一とすること。
  #全体の文字列に対して'ー|一|‐'を'｜'に置き換える
  #何故か半角'-'は置き換えもできない
  reg=u'[ー|一|‐｜-]'
  zen = re.sub(reg,u'｜',zen)
  #文字列に'￥'がある場合、その次の文字を'一'に戻す
  zen = zen.replace('￥｜','一')    

  #改行で分割→リスト
  zenlist = zen.split(u'\n')
  #リストを反転
  zenlist.reverse()
  #リスト要素の最大文字数を調べる
  cmax = 0
  for item in zenlist:
    cmax = len(item) if len(item)>cmax else cmax
  #各要素を最大文字数まで右に全角スペースで埋める（.ljust）
  zenstrlist = []
  for item in zenlist:
    zenstrlist.append(item.ljust(cmax,u'　'))
  #各要素の同順の文字を連結してリスト作成(zip使わず)
  tatelist = []
  for i in range(cmax):
    strsub = ''
    for item in zenstrlist:
      strsub += item[i]
    tatelist.append(strsub)
  #リストの各要素を改行で連結
  return u'\n'.join(tatelist)

#-----------------------------------------------------------------
#日付文字列を受けて縦書用（数字を半角組文字）の文字列を返す
#　非数字は縦書、数字は年月日単位で一組に
#日付文字列＝元号99年99月99日、9999年99月99日
#上記の書式以外は通常の全角縦書を返す
#----------------------------------------------------------------
def tatedatestr(datestr):
  #前後の空行などを削除
  sdate = datestr.strip()
  #一旦全角に
  sdate = zenhan.h2z(sdate)
  #数字のみを半角に
  sdate = zenhan.z2h(sdate,2)   #2=digit
  #１文字毎に処理
  ret = ''
  dig = ''
  for i in range(len(sdate)):
    if re.match('^[0-9]',sdate[i]):
      dig += sdate[i]
      #Class Sozoku.dateshow()メソッドの西暦表示簡便化に伴い
      #西暦年を２桁で区切って返すように変更ｰｰｰここから 
      if len(dig)>=2:
        ret += '\n'+dig
        dig = ''
        #--------------------------------- ここまで追加
    elif len(dig):
      ret += '\n'+dig
      ret += '\n'+sdate[i]
      dig = ''
    else:
      ret += '\n'+sdate[i]
  if len(dig):
    ret += '\n'+dig
  ret = ret.strip()
  return ret

## ----------------------------------------------------------
## メモテキスト（複数行あり）の文字列を縦書き表示用に置き換える。
## 処理としては、与えられた複数行の文字列を一括全角処理と
## 半角組文字処理を振り分けて結果を返すのみ。
## メモテキストの数字文字は半角組文字処理を原則とする。
## ただし、'##'から次の'##'までは通常の全角処理とする。
## （後の＃＃がない場合は最後まで全角処理）
def memotatenaka(memostr):
  #改行で分割
  listlines = zenhan.z2h(memostr,3).split()
  #各行毎に縦書き処理
  retlst = []
  zenflg = False
  for li in listlines:
    tatestr = ''
    if '##' in li:
      ind = 0
      lmx = len(li)
      while 1:
        fo = li.find('##',ind)
        if fo>ind:
          if not zenflg:
            tatestr += '\n'+ tatedatestr(li[ind:fo])  #組文字処理
            zenflg = True
          else: #zenflg True
            tatestr += '\n'+ zentategaki(li[ind:fo])  #全角処理
            zenflg = False
        elif fo==-1:
          if not zenflg:
            tatestr += '\n'+ tatedatestr(li[ind:])  #組文字処理
          else:
            tatestr += '\n'+ zentategaki(li[ind:])  #全角処理
          break
        else:
          zenflg = False if zenflg else True
        ind = fo + 2
        if ind >=lmx:
          break
      tatestr = tatestr.strip()
    else:
      if zenflg:
        tatestr = zentategaki(li)  #１行　全角処理
      else:
        tatestr = tatedatestr(li) #１行分全部を半角組文字処理
    retlst.append(tatestr)
  return retlst


#-----------------------------------------------
# コールバック
# 事前にこのモジュールのfontlistにフォントリストが生成されていること
#-----------------------------------------------
def select_font(parent):
  insta = FontSelect_dialog(parent,'フォント名選択ダイアログ')
  if len(insta.result):
    if __name__=='__main__':
      print(insta.result)
  return insta.result

#----------------------------------------------------------------
#　ここから直接実行の処理スクリプト
#----------------------------------------------------------------
if __name__=='__main__':
  root = Tk()
  #フォント名リスト生成
  fontlist = get_font_list()
  #フォント名リストにデフォルトフォントが含まれているかどうか
  if is_font_insystem(dlgfntname,fontlist):
    print(dlgfntname+' in system')
  else:
    print(dlgfntname+' NOT in system')

  if plsystem == 'Windows':
    wd = 14
  else:
    wd = 7
  Button(root,text='フォント名選択',width=wd,
         command=lambda par=root:select_font(par)).pack()
  Button(root,text='終了',width=wd,command=root.quit).pack()

  root.mainloop()
  root.destroy()
  
